package com.phlx.anchor_collect_flutter;

import androidx.annotation.NonNull;

import com.uhf.linkage.Linkage;
import com.uhf.structures.InventoryData;
import com.uhf.structures.OnBluetoothListener;
import com.uhf.structures.OnInventoryListener;
import com.uhf.structures.Rfid_Value;

import java.util.HashMap;
import java.util.Map;

import io.flutter.embedding.engine.plugins.FlutterPlugin;
import io.flutter.plugin.common.EventChannel;
import io.flutter.plugin.common.MethodCall;
import io.flutter.plugin.common.MethodChannel;

/**
 * rfid插件
 */
public class RfidPlugin implements FlutterPlugin, MethodChannel.MethodCallHandler
        , OnBluetoothListener, OnInventoryListener {

    private MethodChannel _methodChannel;
    private Linkage link;

    // 事件派发对象
    private EventChannel.EventSink _eventSink;

    private EventChannel.StreamHandler _streamHandler = new EventChannel.StreamHandler() {
        @Override
        public void onListen(Object arguments, EventChannel.EventSink events) {
            _eventSink = events;
        }

        @Override
        public void onCancel(Object arguments) {
            _eventSink = null;
        }
    };

    @Override
    public void onAttachedToEngine(@NonNull FlutterPluginBinding binding) {
        _methodChannel = new MethodChannel(binding.getBinaryMessenger(), "flutter_rfid_plugin");
        _methodChannel.setMethodCallHandler(this);

        EventChannel channel = new EventChannel(binding.getBinaryMessenger(), "flutter_rfid_plugin/event");
        channel.setStreamHandler(_streamHandler);
    }

    @Override
    public void onDetachedFromEngine(@NonNull FlutterPluginBinding binding) {
        _methodChannel.setMethodCallHandler(null);
    }

    @Override
    public void onMethodCall(@NonNull MethodCall call, @NonNull MethodChannel.Result result) {

        switch (call.method) {
            case "init_rfid":
                initModule();
                break;
            case "get_version":
                Rfid_Value version_value = new Rfid_Value();
                byte[] versionData = getLinkage().getVersion(version_value);
                String version = "";
                if (version_value.value == 0) {
                    version = new String(versionData);
                } else {
                    version = "" + version_value.value;
                }
                result.success(version);
                break;
            case "start_inventory":
                /**
                 * 下发盘点指令
                 *
                 * @param mode   盘点模式:0-单次盘点，1-高速盘点，2-低功耗盘点  默认高速盘点
                 * @param status 掩码状态 0-禁用，1-启用
                 */
                getLinkage().startInventory(1, 0);
                break;
            case "stop_inventory":
                getLinkage().stopInventory();
                break;
            case "push_data":
                getLinkage().pushRemoteRFIDData(call.argument("value"));
                break;
        }

    }

    private void initModule() {
        getLinkage().setRFModuleType(2);
        /**设置设备连接方式为蓝牙连接*/
        getLinkage().setRFConnectMode(2);
    }

    public Linkage getLinkage() {
        if (link == null) {
            link = new Linkage();
            link.initRFID();
        }

        return link;
    }

    @Override
    public void getBluetoothData(byte[] bytes) {
        Map<String, byte[]> result = new HashMap<>();
        result.put("BluetoothData", bytes);
        _eventSink.success(result);
    }

    @Override
    public void getInventoryData(InventoryData inventoryData) {
        Map<String, String> result = new HashMap<>();
        String epc = TextUtil.byteToHexString(inventoryData.getEPC_Data(), inventoryData.getEpcLength()).substring(1);
        String tid = TextUtil.byteToHexString(inventoryData.getData(), inventoryData.getDataLength());

        result.put("InventoryData", epc + "," + tid);
        _eventSink.success(result);
    }
}
