import 'dart:async';

import 'package:anchor_collect_flutter/utils/dialog_utils.dart';
import 'package:flutter/material.dart';
import 'package:flutter_reactive_ble/flutter_reactive_ble.dart';
import 'package:get/get.dart';
import 'package:isar/isar.dart';
import 'package:permission_handler/permission_handler.dart';

import '../models/cattle_resume_entity.dart';
import '../models/dict_entity.dart';
import '../models/user_model.dart';

class BleService extends GetxService  {
  static BleService get to => Get.find();

  var isScanning = false.obs;
  final ble = FlutterReactiveBle();
  final deviceList = <DiscoveredDevice>[].obs;

  //判断是否正在扫描
  StreamSubscription<DiscoveredDevice>? scanSubscription;

  //判断设备选中状态
  final selectedDevice = Rx<DiscoveredDevice?>(null);

  //判断设备连接状态
  final deviceStatus = ''.obs;

  Future<BleService> init() async {
    return this;
  }

  // 切换扫描状态
  Future<void> toggleScan() async {
    if (isScanning.value) {
      await scanSubscription?.cancel();
      isScanning.value = false;
    } else {
      await scanDevices();
    }
  }

  // 扫描设备
  Future<void> scanDevices() async {
    // 检查权限
    bool permissionsGranted = await requestBlePermissions();
    if (!permissionsGranted) {
      Get.snackbar('Permissions required', 'Please grant all necessary permissions');
      return;
    }
    // 扫描开始时，设置 isScanning 为 true
    isScanning.value = true;
    deviceList.clear();

    // 扫描设备
    final scanStream = ble.scanForDevices(
      withServices: [],

      //scanMode: ScanMode.balanced,  // 将 scanMode 设置为 balanced 功耗均衡，扫描慢
      scanMode: ScanMode.lowLatency, // 将 scanMode 设置为 lowLatency 功耗大 扫描快
    );

    // 监听扫描结果
    scanSubscription = scanStream.listen((device) {
      // 如果设备名称不为空，且列表中没有该设备，则添加到列表中
      if (device.name != null && device.name!.isNotEmpty) {
        // 如果列表中没有该设备，则添加到列表中
        if (!deviceList.any((element) => element.id == device.id)) {
          deviceList.add(device);
          deviceList.refresh(); // 手动触发 Obx 的更新
        }
      }
    });
  }

  // 连接设备
  Future<void> connectToDevice(DiscoveredDevice device, BuildContext context) async {
    selectedDevice.value = device; // 设置当前选中的设备

    await scanSubscription?.cancel(); // 在连接设备之前，停止扫描
    isScanning.value = false; // 设置 isScanning 为 false

    DialogUtils.showLoadingDialog('Connecting to ${device.name}');
    // 显示正在连接的对话框
    // showDialog(
    //   context: context,
    //   barrierDismissible: false,
    //   builder: (BuildContext context) {
    //     return AlertDialog(
    //       title: Text('Connecting to ${device.name}'),
    //       content: CircularProgressIndicator(),
    //     );
    //   },
    // );

    // 连接设备
    final connection = ble.connectToDevice(
      id: device.id,
      servicesWithCharacteristicsToDiscover: {},
    );

    // 监听连接状态
    connection.listen((update) async {
      // 如果连接状态为 connected，则显示连接成功的提示
      if (update.connectionState == DeviceConnectionState.connected) {
        await Future.delayed(Duration(seconds: 1)); // 添加延迟
        deviceStatus.value = 'Connected'; // 更新设备的连接状态
        Get.snackbar('Connected', 'Connected to ${device.name}');
        // Navigator.of(context).pop();
        // 关闭对话框
        DialogUtils.dismissDialog();
      } else {
        deviceStatus.value = 'Disconnected'; // 更新设备的连接状态
      }
    });
  }

  // 检查权限
  Future<bool> requestBlePermissions() async {
    var isLocationGranted = await Permission.locationWhenInUse.request();
    print('checkBlePermissions, isLocationGranted=$isLocationGranted');

    var isBleScanGranted = await Permission.bluetoothScan.request();
    print('checkBlePermissions, isBleScanGranted=$isBleScanGranted');
    //
    var isBleConnectGranted = await Permission.bluetoothConnect.request();
    print('checkBlePermissions, isBleConnectGranted=$isBleConnectGranted');
    //
    var isBleAdvertiseGranted = await Permission.bluetoothAdvertise.request();
    print('checkBlePermissions, isBleAdvertiseGranted=$isBleAdvertiseGranted');

    return isLocationGranted == PermissionStatus.granted &&
        isBleScanGranted == PermissionStatus.granted &&
        isBleConnectGranted == PermissionStatus.granted &&
        isBleAdvertiseGranted == PermissionStatus.granted;
  }
}
